//===========================================================================
// Copyright (C) 1998, 2000-2007, 2010, 2011, 2012, 2013 SINTEF ICT,
// Applied Mathematics, Norway.
//
// This file is part of GoTools
//
// This program is free software; you can redistribute it and/or          
// modify it under the terms of the GNU General Public License            
// as published by the Free Software Foundation version 2 of the License. 
//
// This program is distributed in the hope that it will be useful,        
// but WITHOUT ANY WARRANTY; without even the implied warranty of         
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          
// GNU General Public License for more details.                           
//
// You should have received a copy of the GNU General Public License      
// along with this program; if not, see <http://www.gnu.org/licenses>
//
// Contact information: E-mail: tor.dokken@sintef.no                      
// SINTEF ICT, Department of Applied Mathematics,                         
// P.O. Box 124 Blindern,                                                 
// 0314 Oslo, Norway.                                                     
//
// Other licenses are also available for this software, notably licenses
// for:
// - Building commercial software.                                        
// - Building software whose source code you wish to keep private.        
//===========================================================================

#include "GoTools/topology/tpEdge.h"
#include "GoTools/topology/tpUtils.h"

using namespace Go;

//===========================================================================
tpEdge::tpEdge()
    : next_(0), prev_(0), twin_(0)
//=========================================================================== 
{
}

//===========================================================================
tpEdge::~tpEdge()
//===========================================================================
{
}

//===========================================================================
tpEdge* tpEdge::next()
//===========================================================================
{
    return next_;
}

//===========================================================================
tpEdge* tpEdge::prev()
//===========================================================================
{
    return prev_;
}

//===========================================================================
tpEdge* tpEdge::twin()
//===========================================================================
{
    return twin_;
}

//===========================================================================
void tpEdge::connectAfter(tpEdge* edge)
//===========================================================================
{
    if (edge == 0)
	return;
    next_ = edge->next_;
    prev_ = edge;
    edge->next_ = this;
    if (next_)
	next_->prev_ = this;
}

//===========================================================================
void tpEdge::closeLoop(tpEdge* last)
//===========================================================================
{
    ALWAYS_ERROR_IF(prev_ || last->next_, "Could not close loop");

    prev_ = last;
    last->next_ = this;
}

//===========================================================================
void tpEdge::disconnectThis()
//===========================================================================
{
    if (prev_)
	prev_->next_ = next_;
    if (next_)
	next_->prev_ = prev_;
    next_ = 0;
    prev_ = 0;
}

//===========================================================================
void tpEdge::connectTwin(tpEdge* newtwin, int& status)
//===========================================================================
{
    ALWAYS_ERROR_IF((twin_ || newtwin->twin_) &&
		twin_ != newtwin, // Allow existing twins to connect.
		    "Edge already has a twin.");

    if (!(twin_ && twin_ != newtwin && status == 2))
    {
	twin_ = newtwin;
	newtwin->twin_ = this;
	status = 3;
    }
}

//===========================================================================
void tpEdge::disconnectTwin()
//===========================================================================
{
    ALWAYS_ERROR_IF(twin_ == 0, "Edge has no twin.");
    twin_->twin_ = 0;
    twin_ = 0;
}

//===========================================================================
void tpEdge::adjacentEdges(bool at_start_of_edge,
			    std::vector<tpEdge*>& adjacent,
			    std::vector<bool>& at_start)
//===========================================================================
{
    tpUtils::adjacentEdges<tpEdge>(this, at_start_of_edge, adjacent, at_start);
}

//===========================================================================
tpJointType tpEdge::checkContinuity(tpEdge* nextedge, double neighbour,
					double gap, double bend, double kink) const
//===========================================================================
{
    return tpUtils::checkContinuity<tpEdge>(this, nextedge,
					     neighbour, gap, bend, kink);
}
