//===========================================================================
// Copyright (C) 1998, 2000-2007, 2010, 2011, 2012, 2013 SINTEF ICT,
// Applied Mathematics, Norway.
//
// This file is part of GoTools
//
// This program is free software; you can redistribute it and/or          
// modify it under the terms of the GNU General Public License            
// as published by the Free Software Foundation version 2 of the License. 
//
// This program is distributed in the hope that it will be useful,        
// but WITHOUT ANY WARRANTY; without even the implied warranty of         
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          
// GNU General Public License for more details.                           
//
// You should have received a copy of the GNU General Public License      
// along with this program; if not, see <http://www.gnu.org/licenses>
//
// Contact information: E-mail: tor.dokken@sintef.no                      
// SINTEF ICT, Department of Applied Mathematics,                         
// P.O. Box 124 Blindern,                                                 
// 0314 Oslo, Norway.                                                     
//
// Other licenses are also available for this software, notably licenses
// for:
// - Building commercial software.                                        
// - Building software whose source code you wish to keep private.        
//===========================================================================

#ifndef _FACECONNECTIVITY_H_
#define _FACECONNECTIVITY_H_

#include <vector>

using std::vector;
using std::pair;

namespace Go
{

/// A structure storing the connectivity information between
/// two adjacent faces
template <class edgeType>
struct FaceConnectivity
{
public:
  /// Edge of first face along the common boundary
  edgeType *e1_;
  /// Edge of second face along the common boundary
  edgeType *e2_;
  /// The status is:
  /// 0 : edges join smoothly. \f$G^1\f$.
  /// 1 : edges join, but normals are slightly discontinous. A kink.
  /// 2 : edges join, but the normals are discontinous. \f$G^0\f$.
  /// 3 : edges almost join. A gap.
  /// 4 : edges are totally discontinous.
  /// The minimal tpTopologicalInfo has a one-element status vector
  /// and a two-element parameters vector
  vector<int> status_;
  /// Parameter intervals limiting the areas of the found state of continuity 
  vector< pair<double, double> > parameters_;

  /// Constructor
  FaceConnectivity(edgeType* e1, edgeType *e2)
  {
    e1_ = e1;
    e2_ = e2;
  }

  /// Reset edge info
  void setEdges(edgeType* e1, edgeType *e2)
  {
    e1_ = e1;
    e2_ = e2;
  }

  /// The highest continuity between the two faces
  int BestStatus() const
  {
    int s = 4;
    for (size_t i = 0; i < status_.size(); ++i)
      if (status_[i] < s)
	s = status_[i];
    return s;
  }
  /// The lowest continuity between the two faces
  int WorstStatus() const
  {
    int s = 0;
    for (size_t i = 0; i < status_.size(); ++i)
      if (status_[i] > s) s = status_[i];
    return s;
  }
};

} // namespace Go
#endif  //_FACECONNECTIVITY_H_
