//===========================================================================
// Copyright (C) 1998, 2000-2007, 2010, 2011, 2012, 2013 SINTEF ICT,
// Applied Mathematics, Norway.
//
// This file is part of GoTools
//
// This program is free software; you can redistribute it and/or          
// modify it under the terms of the GNU General Public License            
// as published by the Free Software Foundation version 2 of the License. 
//
// This program is distributed in the hope that it will be useful,        
// but WITHOUT ANY WARRANTY; without even the implied warranty of         
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          
// GNU General Public License for more details.                           
//
// You should have received a copy of the GNU General Public License      
// along with this program; if not, see <http://www.gnu.org/licenses>
//
// Contact information: E-mail: tor.dokken@sintef.no                      
// SINTEF ICT, Department of Applied Mathematics,                         
// P.O. Box 124 Blindern,                                                 
// 0314 Oslo, Norway.                                                     
//
// Other licenses are also available for this software, notably licenses
// for:
// - Building commercial software.                                        
// - Building software whose source code you wish to keep private.        
//===========================================================================

#include "sisl-copyright.h"


#define S1944

#include "sislP.h"

#if defined(SISLNEEDPROTOTYPES)
void
   s1944(double etau[],int ik,int in,int idim,double et[],double ed[],
	     int im,int inlc,int inlr,int inorm,double ea[],double ew1[],
	     int nfirst[],int nlast[],double eb[],double ew2[],
	     int n2sta[],double ec[],int *jstat)
#else
void s1944(etau,ik,in,idim,et,ed,im,inlc,inlr,inorm,ea,ew1,nfirst,nlast,eb,ew2,
	   n2sta,ec,jstat)
   double etau[];
   int ik;
   int in;
   int idim;
   double et[];
   double ed[];
   int im;
   int inlc;
   int inlr;
   int inorm;
   double ea[];
   double ew1[];
   int nfirst[];
   int nlast[];
   double eb[];
   double ew2[];
   int n2sta[];
   double ec[];
   int *jstat;
#endif     
/*
*********************************************************************
* 
* PURPOSE    : To set up the normal equations for solving the linear
*              system ea*x=ed in a weighted least squares sense. The
*              least squares problem stems from the problem of approximating
*              the spline with knot vector et and B-spline coefficients 
*              ed from the subspace generated by the knot vector etau
*              which is a subsequence of et. The aim is to minimize a
*              weighted 2-norm of the coefficients of the difference between
*              a spline from the spline space generated by etau and the
*              given spline in the spline space gernerated by et expressed
*              in the latter space.
* 
* INPUT      : etau   - Real array of length (in+ik) containing the  
*                       knot vector of the approximating space.
*	       ik     - The order of the spline space.
*              in     - The dimension of the spline space corresponding
*                       to etau.
*              idim   - The dimension of the geometry space.
*              et     - Real array of length (im+ik) containing the refined
*                       knot vector.
*              ed     - Real array of length (im*idim) containing the B-spline
*                       coefficients of the spline to be approximated.
*              im     - The dimension of the spline space corresponding to et.
*              inlc   - Number of columns of corner element.
*              inlr   - Number of rows of corner element.
*              inorm  - Number of rows of corner element of normal equations.
*              ea     - Real array of dimension (im*ik) containing 
*                       the B-spline refinement matrix from the knot vector
*                       etau to the knot vector et. This matrix has
*                       dimension im*in but since at most
*                       ik entries are nonzero in each row, it can
*                       be stored in a im*ik array together
*                       with two integer arrays indicating the position
*                       of the first and last nonzero elements in each
*                       row, and a matrix representing corner elements if
*                       the spline is periodic.
*              ew1    - Corner element originating from periodicity.
*                       Size inlr*inlc.
*              nfirst - Integer array of dimension (im) containing 
*                       pointers to the first nonzero element of each row 
*                       of the B-spline refinement matrix from etau to et.
*              nlast  - Integer array of dimension (im) containing 
*                       pointers to the last nonzero element of each row 
*                       of the B-spline refinement matrix from etau to et.
*
* 
* OUTPUT     : eb     - Real array of dimension (in*ik) containing the 
*                       band part of the coefficient matrix of the normal 
*                       equations. This is really a 
*                       in*in matrix but since each row of ea
*                       only has at most ik nonzero elements and eb
*                       essentially is of the form eb = ea(tr)*ea, it is
*                       easy to see that eb is a band matrix of band with
*                       at most ik. Since eb also is symmetric it is enough
*                       to store the elements below and on the diagonal.
*                       This is done by storing the diagonal element of row
*                       i in eb(i*ik+ik-1), and then letting d2sta[i] indicat
*                       the position of the first nonzero element in row i of eb.
*              ew2    - Corner element of the coefficient matrix of the normal
*                       equations. Size inlc*in.
*              n2sta  - Integer array of length (in*idim) containing pointers to
*                       the first nonzero elements of the in rows of eb.
*              ec     - Real array of lengh (in*idim) containing the right
*                       hand side of the normal equations (or really the
*                       idim right hand sides).
*              jstat      - status messages  
*                                         > 0      : warning
*                                         = 0      : ok
*                                         < 0      : error
*             
* 
* METHOD     : 
*
*
* REFERENCES : 
*              
*
* USE        :
*
*-
* CALLS      :   
*
* WRITTEN BY : Vibeke Skytt, SI, 05.92, on the basis of a routine
*              written by Tom Lyche and Knut Moerken, 12.85.
* REWRITTEN AND RENAMED BY : Vibeke Skytt, SINTEF Oslo, 01.95. 
*                            Introduced periodicity.
*
*********************************************************************
*/
{ 
  int ki,kj,kr;
  int kj2;
  int ki1,ki2,kr1,kj1;
  int kik,kih,krh;
  double tw;
  double thelp;
  double *swa = SISL_NULL;
  
  /* Allocate space for a local array of length in to be used during
     multiplication with dtau(-1/2).  */
  
  if ((swa = newarray(in,DOUBLE)) == SISL_NULL)goto err101;
  
  /* Initiate output arrays to zero.  */
  
  for (kj=0; kj<in; kj++) n2sta[kj] = -1;
  memzero(ec,in*idim,DOUBLE);
  memzero(eb,in*ik,DOUBLE);
  if (inorm > 0)
     memzero(ew2,in*inorm,DOUBLE);
  
  /* Determine the normal equations.
     Compute ea(tr)*dt*ea and ea(tr)*dt*ed and store in eb and ec
     respectively. First perform multipliciation involving only the
     band part of the coefficient matrix. */
  
  for (kj=0; kj<im; kj++)
  {
     ki1 = nfirst[kj];
     ki2 = nlast[kj];
     kr1 = ik - ki2 + ki1 - 1;
     tw = (et[kj+ik] - et[kj])/(double)ik;
     kik = ik - 1;
     for (kr = kr1; kr<ik; ki1++, kik--, kr++)
     {
	if (n2sta[ki1] == -1) n2sta[ki1] = kik;
	kih = ik - ki2 + ki1 - 1;
	krh = ik - 1;
	thelp = tw*ea[kj*ik+kr];
	for (ki=ki1; ki<=ki2; kih++, krh--, ki++)
	   eb[ki*ik+krh] += ea[kj*ik+kih]*thelp;
	kih = kr - ik + ki2 + 1;
	for (ki=0; ki<idim; ki++)
	   ec[kih*idim+ki] += ed[kj*idim+ki]*thelp;
     }
  }
  
  /* Permform multipliciations involving the corner elements. */
  
  for (kj=im-inlr, kj2=0; kj<im; kj++, kj2++)
  {
     ki1 = nfirst[kj];
     ki2 = nlast[kj];
     tw = (et[kj+ik] - et[kj])/(double)ik; 
     for (kr=0; kr<inlc; kr++)
     {
	kih = ik - ki2 + ki1 - 1;
	krh = ik-1;
	thelp = tw*ew1[kj2*inlc+kr];
	for (ki=kr; ki<inlc; ki++, krh--)
	{
	   eb[ki*ik+krh] += ew1[kj2*inlc+ki]*thelp;
	   if (krh < n2sta[ki]) n2sta[ki] = krh;
	}
	for (ki=0; ki<idim; ki++)
	   ec[kr*idim+ki] += ed[kj*idim+ki]*thelp;
	for (ki=ki1; ki<=ki2; kih++, krh--, ki++)
	   ew2[(ki-in+inorm)*in+kr] += ea[kj*ik+kih]*thelp;
     }
  }
  
  /* Multiply eb, ew2 and ec by dtau(-1/2).  */
  
  for (ki=0; ki<in; ki++)
    swa[ki] = sqrt((double)ik/(etau[ki+ik]-etau[ki]));
  for (ki=0; ki<in; ki++)
    {
       thelp = swa[ki];
       for (kj=0; kj<idim; kj++)
	 ec[ki*idim+kj] *= thelp;
       for (kj1=n2sta[ki], kih=kj1-ik+ki+1, kj=kj1;
	kj<ik; kih++, kj++)
	 eb[ki*ik+kj] *= swa[kih]*thelp;
    }
  for (ki=0; ki<inorm; ki++)
     for (kj=0; kj<inlc; kj++)
	ew2[ki*in+kj] *= swa[in-inorm+ki]*swa[kj]; 
  
  /* Copy the last inlc rows of the band part of the matrix into ew2. */
  
  for (ki2=0, ki=in-inorm; ki<in; ki2++, ki++)
  {
     for (kj=n2sta[ki]; kj<ik; kj++)
	ew2[ki2*in+ki-ik+kj+1] += eb[ki*ik+kj];
  }

  /* Normal equations set.  */
  
   *jstat = 0;
   goto out;
   
   /* Error in space allocation.  */
   
   err101: *jstat = -101;
   goto out;
   
   out:
      /* Free scratch used for local array.  */
      
      if (swa != SISL_NULL) freearray(swa);
	  
      return;
}
   
