//===========================================================================
// Copyright (C) 1998, 2000-2007, 2010, 2011, 2012, 2013 SINTEF ICT,
// Applied Mathematics, Norway.
//
// This file is part of GoTools
//
// This program is free software; you can redistribute it and/or          
// modify it under the terms of the GNU General Public License            
// as published by the Free Software Foundation version 2 of the License. 
//
// This program is distributed in the hope that it will be useful,        
// but WITHOUT ANY WARRANTY; without even the implied warranty of         
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          
// GNU General Public License for more details.                           
//
// You should have received a copy of the GNU General Public License      
// along with this program; if not, see <http://www.gnu.org/licenses>
//
// Contact information: E-mail: tor.dokken@sintef.no                      
// SINTEF ICT, Department of Applied Mathematics,                         
// P.O. Box 124 Blindern,                                                 
// 0314 Oslo, Norway.                                                     
//
// Other licenses are also available for this software, notably licenses
// for:
// - Building commercial software.                                        
// - Building software whose source code you wish to keep private.        
//===========================================================================

#include "GoTools/parametrization/PrFaber_F.h"

//-----------------------------------------------------------------------------
void
PrFaber_F::decompose(int jlev, int dim)
//-----------------------------------------------------------------------------
// The nodes in the triangulation belonging to V^j
// form a piecewise linear function f^j in the space S^j.
// This routines decomposes f^j
// into f^{j-1} + g^{j-1} where
// f^{j-1} \in S^{j-1} and g^{j-1} \in W^{j-1}.
{
  if(dim != 1 && dim != 3) return;
  if(jlev < 1 || jlev > t_->getFinestLevel()) return;

  int nprev = t_->getNumNodes(jlev-1);

  double newval;
  for(int i=0; i< nprev; i++)
  {
    t_->getNeighbours(i,jlev,neighbours_);
    for(size_t j=0; j< neighbours_.size(); j++)
    {
      int jj = neighbours_[j];
      if(dim == 3)
      {
        newval = t_->getX(jj) - 0.5 * t_->getX(i);
        t_->setX(jj,newval);
        newval = t_->getY(jj) - 0.5 * t_->getY(i);
        t_->setY(jj,newval);
      }
      newval = t_->getZ(jj) - 0.5 * t_->getZ(i);
      t_->setZ(jj,newval);
    }
  }
}

//-----------------------------------------------------------------------------
void
PrFaber_F::compose(int jlev, int dim)
//-----------------------------------------------------------------------------
// The nodes in the triangulation belonging to V^j
// form a piecewise linear function f^j in the space S^j.
// This routines composes f^j
// from f^{j-1} + g^{j-1} where
// f^{j-1} \in S^{j-1} and g^{j-1} \in W^{j-1}.
{
  if(dim != 1 && dim != 3) return;
  if(jlev < 1 || jlev > t_->getFinestLevel()) return;

  int nprev = t_->getNumNodes(jlev-1);

  double newval;
  for(int i=0; i< nprev; i++)
  {
    t_->getNeighbours(i,jlev,neighbours_);
    for(size_t j=0; j< neighbours_.size(); j++)
    {
      int jj = neighbours_[j];
      if(dim == 3)
      {
        newval = t_->getX(jj) + 0.5 * t_->getX(i);
        t_->setX(jj,newval);
        newval = t_->getY(jj) + 0.5 * t_->getY(i);
        t_->setY(jj,newval);
      }
      newval = t_->getZ(jj) + 0.5 * t_->getZ(i);
      t_->setZ(jj,newval);
    }
  }
}

