//===========================================================================
// Copyright (C) 1998, 2000-2007, 2010, 2011, 2012, 2013 SINTEF ICT,
// Applied Mathematics, Norway.
//
// This file is part of GoTools
//
// This program is free software; you can redistribute it and/or          
// modify it under the terms of the GNU General Public License            
// as published by the Free Software Foundation version 2 of the License. 
//
// This program is distributed in the hope that it will be useful,        
// but WITHOUT ANY WARRANTY; without even the implied warranty of         
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          
// GNU General Public License for more details.                           
//
// You should have received a copy of the GNU General Public License      
// along with this program; if not, see <http://www.gnu.org/licenses>
//
// Contact information: E-mail: tor.dokken@sintef.no                      
// SINTEF ICT, Department of Applied Mathematics,                         
// P.O. Box 124 Blindern,                                                 
// 0314 Oslo, Norway.                                                     
//
// Other licenses are also available for this software, notably licenses
// for:
// - Building commercial software.                                        
// - Building software whose source code you wish to keep private.        
//===========================================================================

#ifndef PRFABER_F_H
#define PRFABER_F_H

#include "GoTools/parametrization/PrFilterbank.h"

/*<PrFaber_F-syntax: */

/** - Implements algorithms for composing and decomposing piecewise
 * linear functions over nested triangulations, using Faber decomposition.
 * For a comparison with prewavelets, see
       
                   M. S. Floater, E. G. Quak and M. Reimers,
                   Filter Bank Algorithms for Piecewise Linear
                   Prewavelets on Arbitrary Triangulations,
                   J. Comp. Appl. Math. \e 119 (2000), 185--207.
*/

class PrFaber_F : public PrFilterbank
{
private:

public:
  /// Constructor
  PrFaber_F() {}
  /// Destructor
  virtual ~PrFaber_F() {};

  // derived from base class...
                   
  /** The nodes in the triangulation belonging to \f$V^j\f$
   * form a piecewise linear function \f$f^j\f$ in the space \f$S^j\f$
   * (where j = level).
   * This routines decomposes \f$f^j\f$
   * into \f$f^{j-1} + g^{j-1}\f$ where
   * \f$f^{j-1} \in S^{j-1}\f$ and \f$g^{j-1} \in W^{j-1}\f$.
   * Use dim = 1 for explicit triangulations and
   * dim = 3 for non-explicit. Default is dim = 1.
   */
  virtual void       decompose(int jlev, int dim = 1);
                       // 1 <= jlev <= finest level

  /** The nodes in the triangulation belonging to \f$V^j\f$
   * form a piecewise linear function \f$f^j\f$ in the space \f$S^j\f$.
   * This routines composes \f$f^j\f$
   * from \f$f^{j-1}\f$ + \f$g^{j-1}\f$ where
   * \f$f^{j-1} \in S^{j-1}\f$ and \f$g^{j-1} \in W^{j-1}\f$.
   * Use dim = 1 for explicit triangulations and
   * dim = 3 for non-explicit. Default is dim = 1.
   */
  virtual void       compose(int jlev, int dim = 1);
                       // 1 <= jlev <= finest level

};

/*>PrFaber_F-syntax: */

/*Class:PrFaber_F

Name:              PrFaber_F
Syntax:	           @PrFaber_F-syntax
Keywords:
Description:       This class implements algorithms for decomposing
                   and decomposing piecewise linear functions over nested
                   triangulations, using Faber decomposition.
                   For a comparison with prewavelets, see
       
                   M. S. Floater, E. G. Quak and M. Reimers,
                   Filter Bank Algorithms for Piecewise Linear
                   Prewavelets on Arbitrary Triangulations,
                   J. Comp. Appl. Math. {\bf 119} (2000), 185--207.

Member functions:
                   "decompose(int level, int dim)" --\\
                   The nodes in the triangulation belonging to V^j
                   form a piecewise linear function f^j in the space S^j
                   (where j = level).
                   This routines decomposes f^j
                   into f^{j-1} + g^{j-1} where
                   f^{j-1} \in S^{j-1} and g^{j-1} \in W^{j-1}.
                   Use dim = 1 for explicit triangulations and
                   dim = 3 for non-explicit. Default is dim = 1.
                   
                   "compose(int level, int dim)" --\\
                   The nodes in the triangulation belonging to V^j
                   form a piecewise linear function f^j in the space S^j.
                   This routines composes f^j
                   from f^{j-1} + g^{j-1} where
                   f^{j-1} \in S^{j-1} and g^{j-1} \in W^{j-1}.
                   Use dim = 1 for explicit triangulations and
                   dim = 3 for non-explicit. Default is dim = 1.
                   
Constructors:
Files:
Example:

See also:
Developed by:      SINTEF Applied Mathematics, Oslo, Norway
Author:	           Michael Floater, SINTEF
Date:              Dec. 98
*/

#endif // PRFABER_F_H
