//===========================================================================
// Copyright (C) 1998, 2000-2007, 2010, 2011, 2012, 2013 SINTEF ICT,
// Applied Mathematics, Norway.
//
// This file is part of GoTools
//
// This program is free software; you can redistribute it and/or          
// modify it under the terms of the GNU General Public License            
// as published by the Free Software Foundation version 2 of the License. 
//
// This program is distributed in the hope that it will be useful,        
// but WITHOUT ANY WARRANTY; without even the implied warranty of         
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          
// GNU General Public License for more details.                           
//
// You should have received a copy of the GNU General Public License      
// along with this program; if not, see <http://www.gnu.org/licenses>
//
// Contact information: E-mail: tor.dokken@sintef.no                      
// SINTEF ICT, Department of Applied Mathematics,                         
// P.O. Box 124 Blindern,                                                 
// 0314 Oslo, Norway.                                                     
//
// Other licenses are also available for this software, notably licenses
// for:
// - Building commercial software.                                        
// - Building software whose source code you wish to keep private.        
//===========================================================================

#ifndef PRCG_H
#define PRCG_H

#include "GoTools/parametrization/PrMatrix.h"
#include "GoTools/parametrization/PrVec.h"

/*<PrCG-syntax: */

/** PrCG - This class implements the (unpreconditioned) CG method
 * for solving sparse symmetric positive definite linear systems.
 */
class PrCG
{
protected:

  double tolerance_;
  int max_iterations_;

  int it_count_;
  double cpu_time_;
  bool converged_;

public:
  /// Constructor
  PrCG();
  /// Destructor
  ~PrCG() {}

  /// Set the tolerance for the residual.
  void setTolerance(double tolerance = 1.0e-6) {tolerance_ = tolerance;}
  /// Set the maximum number of iterations.
  void setMaxIterations(int max_iterations)
           {max_iterations_ = max_iterations;}
  ///Solve the linear system, replacing the start vector with the solution.
  void solve(const PrMatrix& A, PrVec& x, const PrVec& b);

  /// Get the number of iterations spent for the last call of 'solve()'.
  int getItCount() {return it_count_; }

  /// Get the CPU time spent for the last call of 'solve()'.
  double getCPUTime() {return cpu_time_; }

  /// Check if the last call of 'solve()' managed to converge to a solution.
  bool converged() {return converged_; }
};

/*>PrCG-syntax: */

/*Class:PrCG

Name:              PrCG
Syntax:	           @PrCG-syntax
Keywords:
Description:       This class implements the (unpreconditioned) CG method
                   for solving sparse symmetric positive definite
                   linear systems.
Member functions:
                   "setTolerance()" --\\
                   Set the tolerance for the residual.

                   "setMaxIterations()" --\\
                   Set the maximum number of iterations.

                   "solve(const PrMatrix& A, PrVec& x, const PrVec& b)" --\\
                   Solve the linear system, replacing the start vector
                   with the solution.

Constructors:
Files:
Example:

See also:
Developed by:      SINTEF Applied Mathematics, Oslo, Norway
Author:	           Michael Floater, SINTEF
Date:              Dec. 98
*/

#endif // PRCG_H
