//===========================================================================
// Copyright (C) 1998, 2000-2007, 2010, 2011, 2012, 2013 SINTEF ICT,
// Applied Mathematics, Norway.
//
// This file is part of GoTools
//
// This program is free software; you can redistribute it and/or          
// modify it under the terms of the GNU General Public License            
// as published by the Free Software Foundation version 2 of the License. 
//
// This program is distributed in the hope that it will be useful,        
// but WITHOUT ANY WARRANTY; without even the implied warranty of         
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          
// GNU General Public License for more details.                           
//
// You should have received a copy of the GNU General Public License      
// along with this program; if not, see <http://www.gnu.org/licenses>
//
// Contact information: E-mail: tor.dokken@sintef.no                      
// SINTEF ICT, Department of Applied Mathematics,                         
// P.O. Box 124 Blindern,                                                 
// 0314 Oslo, Norway.                                                     
//
// Other licenses are also available for this software, notably licenses
// for:
// - Building commercial software.                                        
// - Building software whose source code you wish to keep private.        
//===========================================================================

#ifndef _IMPLICITUTILS_H
#define _IMPLICITUTILS_H


#include "GoTools/utils/BaryCoordSystem.h"
#include "GoTools/geometry/PointCloud.h"
#include <vector>


namespace Go {


class BernsteinPoly;
class BernsteinMulti;
class BernsteinTriangularPoly;
class BernsteinTetrahedralPoly;
class SplineCurve;
class SplineSurface;
class BoundingBox;


/// Creates a barycentric coordinate system from a given spline curve
void create_bary_coord_system2D(const SplineCurve& curve,
				BaryCoordSystem2D& bc);

/// Creates a barycentric coordinate system from a given 3D spline curve
void create_bary_coord_system3D(const SplineCurve& curve,
				BaryCoordSystem3D& bc);

/// Creates a barycentric coordinate system from a given spline surface
void create_bary_coord_system3D(const SplineSurface& surface,
				BaryCoordSystem3D& bc);

/// Creates a barycentric coordinate system from a point cloud
void create_bary_coord_system3D(const PointCloud3D& cloud,
				BaryCoordSystem3D& bc);

/// Creates a barycentric coordinate system from a bounding box
void create_bary_coord_system3D(const BoundingBox& box,
				BaryCoordSystem3D& bc);

/// Creates a new curve with control points in 3 barycentric coordinates
/// from the 2D input curve.
void cart_to_bary(const SplineCurve& cv, const BaryCoordSystem2D& bc,
		  SplineCurve& cv_bc);

/// Creates a new curve with control points in 4 barycentric coordinates
/// from the 3D input curve.
void cart_to_bary(const SplineCurve& cv, const BaryCoordSystem3D& bc,
		  SplineCurve& cv_bc);

/// Creates a new surface with control points in 4 barycentric coordinates
/// from the 3D input surface.
void cart_to_bary(const SplineSurface& sf, const BaryCoordSystem3D& bc,
		  SplineSurface& sf_bc);

/// Creates a new point cloud in 4 barycentric coordinates
/// from the 3D input cloud.
void cart_to_bary(const PointCloud3D& cloud, const BaryCoordSystem3D& bc,
		  PointCloud4D& cloud_bc);

/// Make the matrix D
void make_matrix(const SplineCurve& curve, int deg,
		 std::vector<std::vector<double> >& mat);

/// Make the matrix D
void make_matrix(const SplineSurface& surf, int deg,
		 std::vector<std::vector<double> >& mat);

/// Make the matrix D
void make_matrix(const PointCloud4D& cloud, int deg,
		 std::vector<std::vector<double> >& mat);

/// Performs implicitization using SVD. This method is suitable when
/// the implicitization is approximate. If the implicitization is
/// exact, make_implicit_gauss() is better. Based on the function
/// SVD() from the newmat matrix library.
void make_implicit_svd(std::vector<std::vector<double> >& mat, 
		       std::vector<double>& b, double& sigma_min);

/// Performs implicitization using Gaussian elimination. This method
/// is suitable when the implicitization is exact. If the
/// implicitization is approximate, make_implicit_svd() is better.
void make_implicit_gauss(std::vector<std::vector<double> >& mat,
			 std::vector<double>& b);


} // namespace Go


#endif // _IMPLICITUTILS_H

