//===========================================================================
// Copyright (C) 1998, 2000-2007, 2010, 2011, 2012, 2013 SINTEF ICT,
// Applied Mathematics, Norway.
//
// This file is part of GoTools
//
// This program is free software; you can redistribute it and/or          
// modify it under the terms of the GNU General Public License            
// as published by the Free Software Foundation version 2 of the License. 
//
// This program is distributed in the hope that it will be useful,        
// but WITHOUT ANY WARRANTY; without even the implied warranty of         
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          
// GNU General Public License for more details.                           
//
// You should have received a copy of the GNU General Public License      
// along with this program; if not, see <http://www.gnu.org/licenses>
//
// Contact information: E-mail: tor.dokken@sintef.no                      
// SINTEF ICT, Department of Applied Mathematics,                         
// P.O. Box 124 Blindern,                                                 
// 0314 Oslo, Norway.                                                     
//
// Other licenses are also available for this software, notably licenses
// for:
// - Building commercial software.                                        
// - Building software whose source code you wish to keep private.        
//===========================================================================

#include "GoTools/geometry/SplineUtils.h"
#include <vector>
//using namespace Go;

void 
Go::SplineUtils::refmatrix(const double *et, int im, int ik, 
			   const double *etau, int in,
			   double *ea, int *nfirst,int *nlast)

/*
*********************************************************************
* 
* PURPOSE    : To compute the B-spline refinement transformation matrix
*              from the spline space generated by the knot vector etau
*              to the refined spline space generated by the refined knot
*              vector et.
* 
* INPUT      : et     - Real array of length (im+ik) containing the refined
*                       knot vector.
*              im     - The dimension of the spline space corresponding to et.
*	       ik     - The order of the spline space.
*              etau   - Real array of length (in+ik) containing the original 
*                       knot vector.
*              in     - The dimension of the spline space corresponding
*                       to etau.
*
* 
* OUTPUT     : ea     - Real array of dimension (im*ik) containing 
*                       the B-spline refinement matrix from the knot vector
*                       etau to the knot vector et. This matrix has
*                       dimension im*in but since at most
*                       ik entries are nonzero in each row, it can
*                       be stored in a im*ik array together
*                       with two integer arrays indicating the position
*                       of the first and last nonzero elements in each
*                       row.
*              nfirst - Integer array of dimension (im) containing 
*                       pointers to the first nonzero element of each row 
*                       of the B-spline refinement matrix from etau to et.
*              nlast  - Integer array of dimension (im) containing 
*                       pointers to the last nonzero element of each row 
*                       of the B-spline refinement matrix from etau to et.
*              jstat      - status messages  
*                                         > 0      : warning
*                                         = 0      : ok
*                                         < 0      : error
*             
* 
* METHOD     : 
*
*
* REFERENCES : 
*              
*
* USE        :
*
*-
* CALLS      :   
*
* WRITTEN BY : Vibeke Skytt, SI, 05.92, on the basis of a routine
*              written by Tom Lyche and Knut Moerken, 12.85.
* REVISED BY : Vibeke Skytt, Sintef, 19/04/2002. Go-ified.
*
*********************************************************************
*/
{
   int kj1,kj2;
   int kih1,kih2;
   int ki,kj,kjh,kkj;
   int kmu;
   int kpl, kfi, kla;

   std::vector<double> sahvec(2*ik, 0.0);
   double *sah = &sahvec[0];
   
   /* The first and last few rows of the refinement matrix ea may contain
      only zeroes. The first task is to find the first and the last
      nonzero rows (pointed at by kj1 and kj2).  First kj2 are determined. */

   for (kmu=in+ik-1; etau[kmu-1] == etau[kmu]; kmu--);
   
   for (kjh=im-1; et[kjh+ik]>etau[kmu]; kjh--);
   
   ki = kjh;
   if (et[kjh+ik] == etau[kmu])
      {
	 for (; et[ki-1+ik] == et[kjh+ik]; ki--);
      }
      
   kj2 = std::min(im-1,ki+in+ik-kmu-1);
   
   /* Determine kj1.  */
       
   for (kmu=0; etau[kmu+1] == etau[kmu]; kmu++);
   
   for (kjh=0; et[kjh]<etau[kmu]; kjh++);
   
   ki = kjh;
   if (et[kjh] == etau[kmu])
      {
	 for (; et[ki+1] == et[kjh]; ki++);
      }
   kj1 = std::max(0,ki-kmu);      
      
//     /*  Set all elements of ea to zero. */
      
//     memzero(ea,im*ik,DOUBLE);
   
   /* Determine the refinement transformation.  */
   /* Indicate that rows 1,2,..,kj1 of ea are zero by setting nfirst>nlast
      for these rows.  */
   
   for (kj=0; kj<kj1; kj++)
     {
	nfirst[kj] = ik;
	nlast[kj] = 0;
     }
   
   /* Similarily for rows kj2+1, ... im-1 of ea.  */
   
   for (kj=kj2+1; kj<im; kj++)
     {
	nfirst[kj] = ik;
	nlast[kj] = 0;
     }
   
   /* Compute rows kj1 to kj2 of ea. */
   
   for (kj=kj1; kj<=kj2; kj++)
     {
       for (; etau[kmu+1] <= et[kj]; kmu++);
       kkj = kj;
       Go::SplineUtils::osloalg(kkj, kmu, ik, in, &kpl, &kfi, &kla,
		   et, etau, sah);
       //  	sh1929(etau,in,ik,kmu,et,im,kkj,sah,&kmuprm,&knu,&kstat);
       //  	if (kstat < 0) goto error;
		       
       /* Get the pointers right.  */
		       
       //         nfirst[kj] = MAX(kmuprm-knu,0);
       //         nlast[kj] = MIN(kmuprm,in-1);
       //         kih1 = nfirst[kj] + ik - kmuprm - 1;
       //         kih2 = nlast[kj] + ik - kmuprm - 1;
       nfirst[kj] = std::max(kfi, 0);
       nlast[kj] = std::min(kla, in-1);
       kih1 = kfi + kpl;
       kih2 = kpl + kla;
       for (ki=kih1; ki<=kih2; ki++)
	 ea[kj*ik+ki] = sah[ki];
     }

   /* Refinement transformation matrix computed.  */


}
