//===========================================================================
// Copyright (C) 1998, 2000-2007, 2010, 2011, 2012, 2013 SINTEF ICT,
// Applied Mathematics, Norway.
//
// This file is part of GoTools
//
// This program is free software; you can redistribute it and/or          
// modify it under the terms of the GNU General Public License            
// as published by the Free Software Foundation version 2 of the License. 
//
// This program is distributed in the hope that it will be useful,        
// but WITHOUT ANY WARRANTY; without even the implied warranty of         
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          
// GNU General Public License for more details.                           
//
// You should have received a copy of the GNU General Public License      
// along with this program; if not, see <http://www.gnu.org/licenses>
//
// Contact information: E-mail: tor.dokken@sintef.no                      
// SINTEF ICT, Department of Applied Mathematics,                         
// P.O. Box 124 Blindern,                                                 
// 0314 Oslo, Norway.                                                     
//
// Other licenses are also available for this software, notably licenses
// for:
// - Building commercial software.                                        
// - Building software whose source code you wish to keep private.        
//===========================================================================

#include "GoTools/geometry/ElementarySurface.h"
#include "GoTools/geometry/SplineSurface.h"


using std::swap;


namespace Go
{


// Constructor
//===========================================================================
ElementarySurface::ElementarySurface()
    : isSwapped_(false)
//===========================================================================
{
}


// Destructor
//===========================================================================
ElementarySurface::~ElementarySurface()
//===========================================================================
{
}


//===========================================================================
CurveLoop ElementarySurface::outerBoundaryLoop(double degenerate_epsilon) const
//===========================================================================
{
  if (isBounded())
    {
      // If surface is bounded, use outerBoundaryLoop on geometrySurface as default
      // Might be overridden in subclasses
      shared_ptr<SplineSurface> asSplineSurface = shared_ptr<SplineSurface>(geometrySurface());
      return asSplineSurface->outerBoundaryLoop(degenerate_epsilon);
    }
  else
    {
      // If surface is unbounded, give an empty loop
      CurveLoop loop;
      return loop;
    }
}


//===========================================================================
RectDomain ElementarySurface::containingDomain() const
//===========================================================================
{
    RectDomain dom = dynamic_cast<const RectDomain&>(parameterDomain());
    return dom;
}


//===========================================================================
bool ElementarySurface::inDomain(double u, double v) const 
//===========================================================================
{
    Array<double, 2> pnt(u, v);
    double eps = 1.0e-8;  // A small number
    return parameterDomain().isInDomain(pnt, eps);
}

//===========================================================================
Point ElementarySurface::closestInDomain(double u, double v) const 
//===========================================================================
{
    Array<double, 2> pnt(u, v);
    Array<double, 2> close(0.0, 0.0);
    double eps = 1.0e-8;  // A small number
    parameterDomain().closestInDomain(pnt, close, eps);
    return Point(close.x(), close.y());
}

//===========================================================================
double ElementarySurface::area(double tol) const
//===========================================================================
{
    return geometrySurface()->area(tol);
}

//===========================================================================
void ElementarySurface::getCornerPoints(std::vector<std::pair<Point,Point> >& corners) const
//===========================================================================
{
  // VSK 0910. Maybe not the best solution in the long run
  geometrySurface()->getCornerPoints(corners);
}

//===========================================================================
SplineSurface* ElementarySurface::asSplineSurface()
//===========================================================================
{
    return createSplineSurface();
}

//===========================================================================
bool ElementarySurface::isBounded() const
//===========================================================================
{
    // Assume unbounded by default
    return false;
}

//===========================================================================
bool ElementarySurface::isClosed(bool& closed_dir_u, bool& closed_dir_v) const
//===========================================================================
{
    // Assume not closed by default
    closed_dir_u = false;
    closed_dir_v = false;
    return false;
}

//===========================================================================
shared_ptr<ElementaryCurve> 
ElementarySurface::getElementaryParamCurve(ElementaryCurve* space_crv, double tol) const 
//===========================================================================
{
  // Default is not simple elementary parameter curve exists
  shared_ptr<ElementaryCurve> dummy;
  return dummy;
}


//===========================================================================
void ElementarySurface::turnOrientation()
//===========================================================================
{
    // Default behaviour is to swap parameter directions.
    swapParameterDirection();
}


//===========================================================================
void ElementarySurface::reverseParameterDirection(bool direction_is_u)
//===========================================================================
{
    MESSAGE("reverseParameterDirection() not implemented.");
}


//===========================================================================
void ElementarySurface::swapParameterDirection()
//===========================================================================
{
    isSwapped_ = !isSwapped_;
}    
    
    
////===========================================================================
//bool ElementarySurface::isReversedU() const
////===========================================================================
//{
//    return isReversedU_;
//}    
    
    
////===========================================================================
//bool ElementarySurface::isReversedV() const
////===========================================================================
//{
//    return isReversedV_;
//}    
    
    
//===========================================================================
bool ElementarySurface::isSwapped() const
//===========================================================================
{
    return isSwapped_;
}    
    
    
//===========================================================================
void ElementarySurface::getOrientedParameters(double& u, double& v) const 
//===========================================================================
{
    if (isSwapped_) {
        swap(u, v);
    }
    //if (isReversedU_) {
    //    // Do something
    //}
    //if (isReversedV_) {
    //    // Do someting
    //}
}



} // namespace Go
