//===========================================================================
// Copyright (C) 1998, 2000-2007, 2010, 2011, 2012, 2013 SINTEF ICT,
// Applied Mathematics, Norway.
//
// This file is part of GoTools
//
// This program is free software; you can redistribute it and/or          
// modify it under the terms of the GNU General Public License            
// as published by the Free Software Foundation version 2 of the License. 
//
// This program is distributed in the hope that it will be useful,        
// but WITHOUT ANY WARRANTY; without even the implied warranty of         
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          
// GNU General Public License for more details.                           
//
// You should have received a copy of the GNU General Public License      
// along with this program; if not, see <http://www.gnu.org/licenses>
//
// Contact information: E-mail: tor.dokken@sintef.no                      
// SINTEF ICT, Department of Applied Mathematics,                         
// P.O. Box 124 Blindern,                                                 
// 0314 Oslo, Norway.                                                     
//
// Other licenses are also available for this software, notably licenses
// for:
// - Building commercial software.                                        
// - Building software whose source code you wish to keep private.        
//===========================================================================

#ifndef STREAM_UTILS_H_
#define STREAM_UTILS_H_

#include <iostream>
#include <vector>

namespace { // anonymous, local namespace
  const char separator = ' ';
};

// =============================================================================
// Generic templates for sending objects to and retreaving objects from stream
// =============================================================================

// Generic write
template <typename T> void object_to_stream(std::ostream&  os, const T& obj) { os << obj << separator;}
template <typename T> void object_to_stream(std::wostream& os, const T& obj) { os << obj << separator;}

// Generic read
template <typename T> void object_from_stream(std::istream&  is, T& obj) { is >> obj; }
template <typename T> void object_from_stream(std::wistream& is, T& obj) { is >> obj; }

// =============================================================================
// SPECIALIZED TEMPLATES FOR CONTAINERS/OTHER PARTICULAR OBJECTS
// =============================================================================

// =============================================================================
// Write specialization for STL vectors
template<typename T>
void object_to_stream(std::ostream& os, const std::vector<T>& v)
// =============================================================================
{ 
  os << v.size() << separator;
  for (auto i = v.begin(); i != v.end(); ++i) object_to_stream(os, *i);
  os << '\n';
}

// =============================================================================
// Read specialization for STL vectors
template<typename T>
void object_from_stream(std::istream& is, std::vector<T>& v)
// =============================================================================
{
  size_t size;
  is >> size;
  v.resize(size);
  for (auto i = v.begin(); i != v.end(); ++i) { object_from_stream(is, *i);}
}

// =============================================================================
// Write specialization for STL vectors
template<typename T>
void object_to_stream(std::wostream& os, const std::vector<T>& v)
// =============================================================================
{ 
  os << v.size() << separator;
  for (auto i = v.begin(); i != v.end(); ++i ) object_to_stream(os, *i);
  os << '\n';
}

// =============================================================================
// Read specialization for STL vectors
template<typename T>
void object_from_stream(std::wistream& is, std::vector<T>& v)
// =============================================================================
{
  size_t size;
  is >> size;
  v.resize(size);
  for (auto i = v.begin(); i != v.end(); ++i) { object_from_stream(is, *i);}
}



#endif
