//===========================================================================
// Copyright (C) 1998, 2000-2007, 2010, 2011, 2012, 2013 SINTEF ICT,
// Applied Mathematics, Norway.
//
// This file is part of GoTools
//
// This program is free software; you can redistribute it and/or          
// modify it under the terms of the GNU General Public License            
// as published by the Free Software Foundation version 2 of the License. 
//
// This program is distributed in the hope that it will be useful,        
// but WITHOUT ANY WARRANTY; without even the implied warranty of         
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          
// GNU General Public License for more details.                           
//
// You should have received a copy of the GNU General Public License      
// along with this program; if not, see <http://www.gnu.org/licenses>
//
// Contact information: E-mail: tor.dokken@sintef.no                      
// SINTEF ICT, Department of Applied Mathematics,                         
// P.O. Box 124 Blindern,                                                 
// 0314 Oslo, Norway.                                                     
//
// Other licenses are also available for this software, notably licenses
// for:
// - Building commercial software.                                        
// - Building software whose source code you wish to keep private.        
//===========================================================================

#ifndef _FTFACEBASE_H
#define _FTFACEBASE_H


#include "GoTools/compositemodel/ftEdgeBase.h"
#include "GoTools/geometry/ParamSurface.h"
#include "GoTools/compositemodel/ftMessage.h"
#include "GoTools/igeslib/ftTangPriority.h"

namespace Go
{


// Forward declarations
class ftEdgeBase;
class ftSurface;

/** ftFaceBase -  An abstract interface to a topological face
 * 
 */
class GO_API ftFaceBase
{
public:
    /// Constructor
    ftFaceBase();
    /// Constructor
    ftFaceBase(int id/*, bool is_turned = false*/);
    /// Destructor
    virtual ~ftFaceBase();

    // Return as type ftSurface
    virtual ftSurface* asFtSurface();

    /// Reset loop information
    virtual void clearInitialEdges()
	{}  // Overriden when required

    // Evaluation and interrogation.
    /// Compute the edges associated to this face or fetch already existing
    /// edges
    virtual std::vector<shared_ptr<ftEdgeBase> > 
      createInitialEdges(double degenerate_epsilon = DEFAULT_SPACE_EPSILON,
			 double kink = 0.00015, bool no_split = false) = 0;
    /// Return pointers to first part of all bd cvs.
    virtual std::vector<shared_ptr<ftEdgeBase> > startEdges() = 0;
    /// Evaluate point on face
    virtual Point point(double u, double v) const = 0;
    /// Evaluate surface normal
    virtual Point normal(double u, double v) const = 0;
    /// The bounding box corresponding to this face
    virtual BoundingBox boundingBox() = 0;
    /// Set id for this face
    virtual void setId(int id);
    /// Return id, default id is -1
    virtual int getId();
    //virtual void turnOrientation() = 0;
    //virtual bool getOrientation() = 0;
    //virtual std::vector<shared_ptr<ftEdgeBase> > 
    //  setOrientation(double degenerate_epsilon=DEFAULT_SPACE_EPSILON) = 0;
    //void turnFace(std::vector<ftFaceBase*>& turned);

    /// Fetch geometric surface
    virtual shared_ptr<ParamSurface> surface() = 0;
    /// Make sure that a geometric surface exists. Not applicable in normal
    /// configurations
    virtual ftMessage createSurf(double& max_error, double& mean_error) = 0;
    /// Fetch the error due to approximations in createSurf. Not applicable in normal
    /// configurations
    virtual void getError(double& max_error, double& mean_error) = 0;
    /// Priority of current face.  Not applicable in normal configurations
    virtual ftTangPriority getPrioType() const = 0;
    /// Update the boundary loops corresponding to this face with a new edge,
    /// i.e. one edge is split and all related topology information must be
    /// updated accordingly
    virtual void updateBoundaryLoops(shared_ptr<ftEdgeBase> new_edge);
    /// Remove all adjacency information related to this face
    virtual void isolateFace()
    {
      // Default no action
      ;
    }

    /// Close gap between adjacent faces
    virtual ftMessage removeGap(ftEdgeBase* e1, ftEdgeBase* e2, ftFaceBase *other)
	{ return FT_NOT_SUPPORTED; }

    /// Closest point between this face and a point
    virtual void closestPoint(const Point& pt,
		      double&  clo_u,
		      double&  clo_v, 
		      Point& clo_pt,
		      double&  clo_dist,
		      double   epsilon) const = 0;


 
protected:
    int id_;
    //bool is_turned_;
};

} // namespace Go


#endif // _FTFACEBASE_H

