//===========================================================================
// Copyright (C) 1998, 2000-2007, 2010, 2011, 2012, 2013 SINTEF ICT,
// Applied Mathematics, Norway.
//
// This file is part of GoTools
//
// This program is free software; you can redistribute it and/or          
// modify it under the terms of the GNU General Public License            
// as published by the Free Software Foundation version 2 of the License. 
//
// This program is distributed in the hope that it will be useful,        
// but WITHOUT ANY WARRANTY; without even the implied warranty of         
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          
// GNU General Public License for more details.                           
//
// You should have received a copy of the GNU General Public License      
// along with this program; if not, see <http://www.gnu.org/licenses>
//
// Contact information: E-mail: tor.dokken@sintef.no                      
// SINTEF ICT, Department of Applied Mathematics,                         
// P.O. Box 124 Blindern,                                                 
// 0314 Oslo, Norway.                                                     
//
// Other licenses are also available for this software, notably licenses
// for:
// - Building commercial software.                                        
// - Building software whose source code you wish to keep private.        
//===========================================================================

#ifndef COMPLETEEDGENET_H
#define COMPLETEEDGENET_H

#include "GoTools/compositemodel/SurfaceModel.h"

namespace Go
{
  /// \brief Complete the edge net of a SurfaceModel. Fetches the wire
  /// frame model corresponding to a surface model, and extends it such
  /// that the extended wire frame will become the wire frame model corresponding
  /// to a volume model have the given surface model as its outer boundary.
  /// The extension to the wireframe is represented as pairs of vertices where
  /// these vertices lie at the endpoints of the missing edges.
  /// NB! This solution is currently not expected to handle all configurations.

  class CompleteEdgeNet
  {
  public:
    /// Constructor. The method is applied on a SurfaceModel
    /// \param sfmodel Pointer to a SurfaceModel
    CompleteEdgeNet(shared_ptr<SurfaceModel> sfmodel, 
		    bool perform_step2);

    /// Destructor
    ~CompleteEdgeNet();

    /// Apply algorithm for completing the edge net and create
    /// a starting ground for a block structured model of solids
    /// \return Whether the edge net was completed or not. 
    bool perform(std::vector<std::pair<Point,Point> >& corr_vx_pts);

    /// Fetch modified model (after regularization)
    /// \return Pointer to the modified model
    shared_ptr<SurfaceModel> getRegularizedModel()
      {
	return model_;
      }

    /// Fetch new edges represented by their end vertices
    /// \return  Vector of pointers to end vertices of the edges
    std::vector<std::pair<shared_ptr<Vertex>, shared_ptr<Vertex> > >
      getMissingEdges()
      {
	return missing_edges_;
      }

  private:
    shared_ptr<SurfaceModel> model_;
    std::vector<std::pair<shared_ptr<Vertex>, shared_ptr<Vertex> > > missing_edges_;
    bool perform_step2_;

    /// Given a regular solid, add the edges required to make a block 
    /// structured model
    void addMissingEdges();

    void traverseEdges(std::vector<ftEdge*>& edges,
		       std::vector<ftEdge*>& curr_path,
		       std::vector<int>& curr_idx,
		       ftEdge *curr_edge,
		       shared_ptr<Vertex> vx,
		       bool search_end); 
 
    ftEdge* fetchNextEdge(ftEdge *curr_edge,
			  shared_ptr<Vertex> vx,
			  int& next_idx);

    bool regularizeEdgeLoop(std::vector<ftEdge*>& edges);

    void splitLoop(std::vector<ftEdge*>& edges,
		   std::vector<shared_ptr<Vertex> >& vxs,
		   bool to_add_edges,
		   std::vector<std::vector<ftEdge*> >& split_loops,
		   std::vector<std::vector<shared_ptr<Vertex> > >& split_vxs,
		   std::vector<bool>& add_edges_split);

    bool regularizeCurrLoop(std::vector<ftEdge*>& edges,
			    std::vector<shared_ptr<Vertex> >& vxs,
			    bool to_add_edges);

    double getVertexAngle(ftEdge *edge1, ftEdge *edge2);

    void addRemainingEdges();

    bool vertexInfo(shared_ptr<Vertex> vx, double& angle, Point& centre);

    void writePath(std::vector<ftEdge*>& edges, shared_ptr<Vertex> vx);

    std::vector<ftEdge*> getStartEdges();

    void addIdentifiedEdges(std::vector<std::pair<Point,Point> >& corr_vx_pts);
  };

} // namespace Go

#endif // COMPLETEEDGENET_H
